/*
 * misc colorspace conversion functions
 *
 * most of them have common arguments (wanna be able to use function
 * pointers):
 *	 foo(unsigned char* dest, unsigned char* src, int width, int height);
 *
 *  (c) 1998 Gerd Knorr <kraxel@goldbach.in-berlin.de>
 *
 */

#include "colorspace.h"
#include "byteorder.h"

/* ------------------------------------------------------------------- */
/* lut stuff                                                           */

static unsigned long   lut_red[256];
static unsigned long   lut_green[256];
static unsigned long   lut_blue[256];

void
lut_init(unsigned long red_mask, unsigned long green_mask,
	 unsigned long blue_mask, int depth, int swap)
{
    int             rgb_red_bits = 0;
    int             rgb_red_shift = 0;
    int             rgb_green_bits = 0;
    int             rgb_green_shift = 0;
    int             rgb_blue_bits = 0;
    int             rgb_blue_shift = 0;
    unsigned int    i;
    unsigned int    mask;

    for (i = 0; i < 32; i++) {
        mask = (1 << i);
        if (red_mask & mask)
            rgb_red_bits++;
        else if (!rgb_red_bits)
            rgb_red_shift++;
        if (green_mask & mask)
            rgb_green_bits++;
        else if (!rgb_green_bits)
            rgb_green_shift++;
        if (blue_mask & mask)
            rgb_blue_bits++;
        else if (!rgb_blue_bits)
            rgb_blue_shift++;
    }
#if 0
    printf("color: bits shift\n");
    printf("red  : %04i %05i\n", rgb_red_bits, rgb_red_shift);
    printf("green: %04i %05i\n", rgb_green_bits, rgb_green_shift);
    printf("blue : %04i %05i\n", rgb_blue_bits, rgb_blue_shift);
#endif
    
    if (rgb_red_bits > 8)
	for (i = 0; i < 256; i++)
	    lut_red[i] = (i << (rgb_red_bits + rgb_red_shift - 8));
    else
	for (i = 0; i < 256; i++)
	    lut_red[i] = (i >> (8 - rgb_red_bits)) << rgb_red_shift;
    
    if (rgb_green_bits > 8)
	for (i = 0; i < 256; i++)
	    lut_green[i] = (i << (rgb_green_bits + rgb_green_shift - 8));
    else
	for (i = 0; i < 256; i++)
	    lut_green[i] = (i >> (8 - rgb_green_bits)) << rgb_green_shift;
    
    if (rgb_blue_bits > 8)
	for (i = 0; i < 256; i++)
	    lut_blue[i] = (i << (rgb_blue_bits + rgb_blue_shift - 8));
    else
	for (i = 0; i < 256; i++)
	    lut_blue[i] = (i >> (8 - rgb_blue_bits)) << rgb_blue_shift;

    if (2 == depth && swap) {
	for (i = 0; i < 256; i++) {
	    lut_red[i] = SWAP2(lut_red[i]);
	    lut_green[i] = SWAP2(lut_green[i]);
	    lut_blue[i] = SWAP2(lut_blue[i]);
	}
    }
    if (4 == depth && swap) {
	for (i = 0; i < 256; i++) {
	    lut_red[i] = SWAP4(lut_red[i]);
	    lut_green[i] = SWAP4(lut_green[i]);
	    lut_blue[i] = SWAP4(lut_blue[i]);
	}
    }
}

void
rgb24_to_lut2(unsigned char *dest, unsigned char *src, int w, int h)
{
    unsigned short *d = (unsigned short*)dest;
    int n = w*h;

    while (n-- > 0) {
	*(d++) = lut_red[src[0]] | lut_green[src[1]] | lut_blue[src[2]];
	src += 3;
    }
}

void
bgr24_to_lut2(unsigned char *dest, unsigned char *src, int w, int h)
{
    unsigned short *d = (unsigned short*)dest;
    int n = w*h;

    while (n-- > 0) {
	*(d++) = lut_red[src[2]] | lut_green[src[1]] | lut_blue[src[0]];
	src += 3;
    }
}

void
rgb32_to_lut2(unsigned char *dest, unsigned char *src, int w, int h)
{
    unsigned short *d = (unsigned short*)dest;
    int n = w*h;

    while (n-- > 0) {
	*(d++) = lut_red[src[1]] | lut_green[src[2]] | lut_blue[src[3]];
	src += 4;
    }
}

void
bgr32_to_lut2(unsigned char *dest, unsigned char *src, int w, int h)
{
    unsigned short *d = (unsigned short*)dest;
    int n = w*h;

    while (n-- > 0) {
	*(d++) = lut_red[src[3]] | lut_green[src[2]] | lut_blue[src[1]];
	src += 4;
    }
}

void
gray_to_lut2(unsigned char *dest, unsigned char *src, int w, int h)
{
    unsigned short *d = (unsigned short*)dest;
    int n = w*h;

    while (n-- > 0) {
	*(d++) = lut_red[*src] | lut_green[*src] | lut_blue[*src];
	src++;
    }
}

void
rgb24_to_lut4(unsigned char *dest, unsigned char *src, int w, int h)
{
    unsigned int *d = (unsigned int*)dest;
    int n = w*h;

    while (n-- > 0) {
	*(d++) = lut_red[src[0]] | lut_green[src[1]] | lut_blue[src[2]];
	src += 3;
    }
}

void
bgr24_to_lut4(unsigned char *dest, unsigned char *src, int w, int h)
{
    unsigned int *d = (unsigned int*)dest;
    int n = w*h;

    while (n-- > 0) {
	*(d++) = lut_red[src[2]] | lut_green[src[1]] | lut_blue[src[0]];
	src += 3;
    }
}

void
rgb32_to_lut4(unsigned char *dest, unsigned char *src, int w, int h)
{
    unsigned int *d = (unsigned int*)dest;
    int n = w*h;

    while (n-- > 0) {
	*(d++) = lut_red[src[1]] | lut_green[src[2]] | lut_blue[src[3]];
	src += 4;
    }
}

void
bgr32_to_lut4(unsigned char *dest, unsigned char *src, int w, int h)
{
    unsigned int *d = (unsigned int*)dest;
    int n = w*h;

    while (n-- > 0) {
	*(d++) = lut_red[src[3]] | lut_green[src[2]] | lut_blue[src[1]];
	src += 4;
    }
}

void
gray_to_lut4(unsigned char *dest, unsigned char *src, int w, int h)
{
    unsigned int *d = (unsigned int*)dest;
    int n = w*h;

    while (n-- > 0) {
	*(d++) = lut_red[*src] | lut_green[*src] | lut_blue[*src];
	src++;
    }
}

/* ------------------------------------------------------------------- */
/* RGB conversions                                                     */

void
rgb24_to_bgr24(unsigned char *dest, unsigned char *src, int w, int h)
{
    register char *s = src;
    register char *d = dest;
    register int   i = w*h;

    while (--i) {
	*(d++) = s[2];
	*(d++) = s[1];
	*(d++) = s[0];
	s += 3;
    }
}

void
bgr24_to_bgr32(unsigned char *dest, unsigned char *src, int w, int h)
{
    int             i    = w*h;

    while (i--) {
        *(dest++) = *(src++);
        *(dest++) = *(src++);
        *(dest++) = *(src++);
	*(dest++) = 0;
    }
}

void
bgr24_to_rgb32(unsigned char *dest, unsigned char *src, int w, int h)
{
    int             i    = w*h;

    while (i--) {
	*(dest++) = 0;
        *(dest++) = src[2];
        *(dest++) = src[1];
        *(dest++) = src[0];
	src +=3;
    }
}

void
rgb32_to_rgb24(unsigned char *dest, unsigned char *src, int w, int h)
{
    int             i    = w*h;

    while (i--) {
	src++;
	*(dest++) = *(src++);
	*(dest++) = *(src++);
	*(dest++) = *(src++);
    }
}

/* ------------------------------------------------------------------- */
/* YUV conversions                                                     */

void
packed422_to_planar422(unsigned char *d, unsigned char *s, int w, int h)
{
    int i;
    unsigned char *y,*u,*v;

    i = (w * h)/2;
    y = d;
    u = y + w * h;
    v = u + w * h / 2;
    
    while (--i) {
	*(y++) = *(s++);
	*(u++) = *(s++);
	*(y++) = *(s++);
        *(v++) = *(s++);
    }
}

void
packed422_to_planar411(unsigned char *d, unsigned char *s, int w, int h)
{
    int  a,b;
    unsigned char *y,*u,*v;

    y = d;
    u = y + w * h;
    v = u + w * h / 4;

    for (a = h; a > 0; a -= 2) {
	for (b = w; b > 0; b -= 2) {
	    
	    *(y++) = *(s++);
	    *(u++) = *(s++);
	    *(y++) = *(s++);
	    *(v++) = *(s++);
	}
	for (b = w; b > 0; b -= 2) {
	    *(y++) = *(s++);
	    s++;
	    *(y++) = *(s++);
	    s++;
	}
    }
}
